% Two SSB-ASK signals are multiplexed on the same optical carrier and demodulated using coherent detection
%
clear all;
global T n NTS NSYM time tindex freq bindex
%
c0=2.9979e8;        % Light speed in vacuum
Lambda=1550e-9;     % Vacuum wavelength of the optical carrier
f0=c0/Lambda;       % Frequency of the optical carrier
T=1/f0;             % Period of the optical carrier
%
%____________________________ FFT PARAMETERS ______________________________
%
NSYM=2048;       % Frequency resolution: df=f0/(2*NSYM)
NTS=1024;           % Time resolution: dt=T/NTS
%
% Time and Frequency Parameters Assignment
%
NTR=24;           % Number of unity intervals to the right of pulse center
NTL=24;           % Number of unity intervals to the left of pulse center
NB=3;               % Multiples of the optical carrier frequency plotted in each semi-axis
freq=(-NTS/2:1/(2*NSYM):(NTS-1/NSYM)/2); % Normalized frequency axis. Normalization f0=1/T
findex=(NTS*NSYM+1:NTS*NSYM+1+NB*2*NSYM);
fplot=freq(findex)/T/1e12; % Unilateral dimensional frequency axis in Thz for plotting functions
bindex=(NTS*NSYM+1-NB*2*NSYM:NTS*NSYM+1+NB*2*NSYM);
bplot=freq(bindex)/T/1e12; % Bilateral dimensional frequency axis in THz for plotting functions
time=(-NSYM:1/NTS:NSYM-1/NTS); % Normalized time axis. Normalization T
tindex=(NTS*NSYM+1-NTL*NTS:NTS*NSYM+1+NTR*NTS); % Temporal axis for pulse representation. Pulse is zero outside
tplot=time(tindex)*T*1e15; % Dimensional temporal axis in fs for plotting output pulse
%
%_________________________________ SIGNALS ________________________________
%
n=20; % Ratio between the carrier frequency and the cutoff frequency of the Gaussian spectrum
I=cos(2*pi*time); % In-phase carrier (cosine function at the unit frequency)
Q=sin(2*pi*time); % Quadrature carrier (sine function at the unit frequency)
H=-1i*sign(freq); % Hilbert filter
%
P1=input('Select first channel reference pulse (S, G, R, B, W) [null]: ','s');
P2=input('Select second channel reference pulse (S, G, R, B, W) [null]: ','s');
r1=SWITCH_PULSE(P1);
r2=SWITCH_PULSE(P2);
%
%----------------------- Channel 1 ASK-SSB Modulation ---------------------
%
% Generation of the harmonic conjugate x1(t) of r1(t)
%
R1=SPECTRUM(r1); % Spectrum of the reference function
X1=R1.*H; % Spectrum of the harmonic conjugate x(t) of r(t)
x1=PULSE(X1); % Harmonic conjugate
%
% Generation of upper and lower SSB signal with reference r1(t)
%
f1_L=r1.*I+x1.*Q; % Time domain modulated signal with LSB
f1_U=r1.*I-x1.*Q; % Time domain modulated signal with USB
F1_L=SPECTRUM(f1_L); % LSB spectrum
F1_U=SPECTRUM(f1_U); % USB spectrum
%
% Channel 1 SSB Demodulation
%
F1_L_DEMOD=SPECTRUM(f1_L.*I); % Demodulated time domain signal from channel 1 LSB spectrum
F1_U_DEMOD=SPECTRUM(f1_U.*I); % Demodulated time domain signal from channel 1 USB spectrum
%
%---------------------- Channel 2 ASK-SSB Modulation ----------------------
%
% Generation of the harmonic conjugate x2(t) of r2(t)
%
R2=SPECTRUM(r2); % Spectrum of the reference function
X2=R2.*H; % Spectrum of the harmonic conjugate x2(t) of r2(t)
x2=PULSE(X2); % Harmonic conjugate
%
% Generation of upper and lower SSB signal with reference r1(t)
%
f2_L=r2.*I+x2.*Q; % Time domain modulated signal with LSB
f2_U=r2.*I-x2.*Q; % Time domain modulated signal with USB
F2_L=SPECTRUM(f2_L); % LSB spectrum
F2_U=SPECTRUM(f2_U); % USB spectrum
%
% Channel 2 SSB Demodulation
%
F2_L_DEMOD=SPECTRUM(f2_L.*I); % Demodulated time domain signal from channel 2 LSB spectrum
F2_U_DEMOD=SPECTRUM(f2_U.*I); % Demodulated time domain signal from channel 2 USB spectrum
%
%------------------------- MUX of SSB-ASK channels ------------------------
%
f1L2U=f1_L+f2_U; % Channel 1 LSB + Channel 2 USB
f1U2L=f1_U+f2_L; % Channel 1 USB + Channel 2 LSB
F1L2U=SPECTRUM(f1L2U); % Channel 1 LSB + Channel 2 USB
F1U2L=SPECTRUM(f1U2L); % Channel 1 USB + Channel 2 LSB
%
%------------------------ DEMUX of SSB-ASK channels -----------------------
%
% Channel 1 LSB + Channel 2 USB --> R1L, R2U
%
U1L2U=SPECTRUM(f1L2U/2.*I); % Factor 1/2 accounts for signal split between the I/Q mixers
V1L2U=SPECTRUM(f1L2U/2.*Q); % Factor 1/2 accounts for signal split between the I/Q mixers
W1L2U=V1L2U.*H;
R1L_DEMOD=U1L2U-W1L2U;
R1L_DEMOD(NTS*NSYM+1)=(R1L_DEMOD(NTS*NSYM)+R1L_DEMOD(NTS*NSYM+2))/2;
R2U_DEMOD=U1L2U+W1L2U;
R2U_DEMOD(NTS*NSYM+1)=(R2U_DEMOD(NTS*NSYM)+R2U_DEMOD(NTS*NSYM+2))/2;
%
% Channel 1 USB + Channel 2 LSB --> R1U, R2L
%
U1U2L=SPECTRUM(f1U2L/2.*I); % Factor 1/2 accounts for signal split between the I/Q mixers
V1U2L=SPECTRUM(f1U2L/2.*Q); % Factor 1/2 accounts for signal split between the I/Q mixers
W1U2L=V1U2L.*H;
R1U_DEMOD=U1U2L+W1U2L;
R1U_DEMOD(NTS*NSYM+1)=(R1U_DEMOD(NTS*NSYM)+R1U_DEMOD(NTS*NSYM+2))/2;
R2L_DEMOD=U1U2L-W1U2L;
R2L_DEMOD(NTS*NSYM+1)=(R2L_DEMOD(NTS*NSYM)+R2L_DEMOD(NTS*NSYM+2))/2;
%
%----------------------------- GRAPHICS -----------------------------------
%
 GRAPH_ASK(r1,x1,R1,f1_L,F1_L,F1_L_DEMOD,f1_U,F1_U,F1_U_DEMOD,...
    r2,x2,R2,f2_L,F2_L,F2_L_DEMOD,f2_U,F2_U,F2_U_DEMOD,...
    f1L2U,F1L2U,R1L_DEMOD,R2U_DEMOD,f1U2L,F1U2L,R1U_DEMOD,R2L_DEMOD);
